<?php

namespace Modules\Domain\Modules\User\Repositories;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Modules\Domain\Modules\SystemConfiguration\Helpers\SystemConfigurationHelper;
use Modules\Domain\Modules\SystemConfiguration\Models\SystemConfiguration;
use Modules\Domain\Modules\User\Enums\RoleEnum;
use Modules\Domain\Modules\User\Helpers\RolHelper;
use Modules\Domain\Modules\User\Helpers\UserHelper;
use Modules\Domain\Modules\User\Models\Person;
use Modules\Domain\Modules\User\Models\Rol;
use Modules\Domain\Modules\User\Models\User;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Modules\Shared\Enum\EmailBodyTemplate;
use Modules\Shared\Services\MailerService;

class UserRepository
{
    public static function params()
    {
        $documents = Person::select('document_type')
            ->distinct()
            ->pluck('document_type')
            ->toArray();

        $roles = Rol::select('id', 'name')
            ->whereNotIn('id', [
                RoleEnum::COMPANY,
            ])
            ->get();

        $result = [
            'documents' => $documents,
            'roles' => $roles,
        ];

        return $result;
    }

    public static function list(Request $request)
    {
        RolHelper::validateAdminAccess();

        UserHelper::validateListRequest($request);

        $staffs = in_array($request->rol_id, [RoleEnum::TEACHER]);
        $students = in_array($request->rol_id, [RoleEnum::STUDENT]);

        $users = User::select([
            'user.id',
            'rol.id as rol_id',
            'user.avatar as photo',
            'person.names',
            'person.phone',
            'person.document_type',
            'person.document_number',
            'user.email',
            'user.is_active',
            'user.last_login',
        ])
            ->join('rol_user', 'user.id', 'rol_user.user_id')
            ->join('rol', function ($join) use ($request) {
                $join->on('rol_user.rol_id', 'rol.id')
                    ->whereIn('rol.id', $request->rol_ids);
            })
            ->join('person', 'person.id', 'user.person_id')
            ->when($staffs, function ($query) {
                $query->join('staff', 'person.id', 'staff.person_id');
            })
            ->when($students, function ($query) {
                $query->join('student', 'person.id', 'student.person_id');
            })
            ->when($request->filled('search'), function ($query) use ($request) {
                $query->where(function ($query) use ($request) {
                    $query->where('person.names', 'like', "%$request->search%")
                        ->orWhere('person.phone', 'like', "%$request->search%")
                        ->orWhere('person.email', 'like', "%$request->search%");
                });
            })
            ->orderBy('person.names')
            ->distinct()
            ->paginate($request->size, ['*'], 'page', $request->page);

        $result = [
            'page' => $request->page,
            'size' => $request->size,
            'total' => $users->total(),
            'users' => $users->items(),
        ];

        return $result;
    }

    public static function create(Request $request)
    {
        RolHelper::validateAdminAccess();

        UserHelper::validateCreateRequest($request);

        $person = Person::create([
            'document_type' => $request->document_type,
            'document_number' => $request->document_number,
            'names' => $request->names,
            'phone' => $request->phone,
            'email' => $request->email,
        ]);

        $user = User::create([
            'person_id' => $person->id,
            'email' => $request->email,
            'password' => Hash::make($request->document_number),
        ]);

        $user->roles()->attach($request->role_id);

        return 'Usuario creado correctamente';
    }

    public static function update(int $id, Request $request)
    {
        $user = UserHelper::validateAccess($id);

        $roles = $user->roles();

        $is_admin = $roles
            ->where('rol.id', RoleEnum::ADMINISTRADOR)
            ->exists();

        UserHelper::validateUpdateRequest($request, $user, $is_admin);

        $columns = $is_admin ? [
            'document_type' => $request->document_type,
            'document_number' => $request->document_number,
            'names' => $request->names,
            'phone' => $request->phone,
            'email' => $request->email,
        ] : [
            'phone' => $request->phone,
            'email' => $request->email,
        ];

        $user->person->update($columns);

        $user->update([
            'email' => $request->email,
        ]);

        return 'Usuario actualizado correctamente';
    }

    public static function changePassword(int $id, Request $request)
    {
        $user = UserHelper::validateAccess($id);

        UserHelper::validateChangePasswordRequest($request);

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return 'Contraseña actualizada correctamente';
    }

    public static function changePhoto(int $id, Request $request)
    {
        $user = UserHelper::validateAccess($id);

        UserHelper::validateChangePhotoRequest($request);

        if ($user->avatar) {
            $user->avatar = 'public/' . $user->avatar;

            if (Storage::exists($user->avatar)) {
                Storage::delete($user->avatar);
            }
        }

        $file = $request->file('file');
        $path = $file->store('public/user');
        $path = str_replace('public/', '', $path);

        $user->update([
            'avatar' => $path,
        ]);

        return $path;
    }

    public static function deletePhoto(int $id)
    {
        $user = UserHelper::validateAccess($id);

        if ($user->avatar) {
            $user->avatar = 'public/' . $user->avatar;

            if (Storage::exists($user->avatar)) {
                Storage::delete($user->avatar);
            }
        }

        $user->update([
            'avatar' => null,
        ]);

        return "Foto eliminada correctamente";
    }

    public static function resetPassword(int $id)
    {
        $user = UserHelper::validateAccess($id);

        $user->update([
            'password' => Hash::make($user->person->document_number),
        ]);

        return 'Contraseña restablecida correctamente';
    }

    public static function disable(int $id)
    {
        $user = UserHelper::validateDisableOrDeleteAccess($id);

        $user->update([
            'is_active' => !$user->is_active,
        ]);

        return "Usuario " . ($user->is_active ? "habilitado" : "deshabilitado") . " correctamente";
    }

    public static function delete(int $id)
    {
        $user = UserHelper::validateDisableOrDeleteAccess($id);

        $roles = $user->roles();

        $is_admin = $roles
            ->where('rol.id', RoleEnum::ADMINISTRADOR)
            ->exists();

        if (!$is_admin) {
            throw new Exception('Solo se pueden eliminar usuarios administradores');
        }

        $count = $roles->count();
        if ($count > 1) {
            $roles->detach(RoleEnum::ADMINISTRADOR);

            return "Usuario eliminado como administrador correctamente";
        }

        $roles->detach();
        $user->delete();

        return "Usuario eliminado correctamente";
    }

    public static function import(Request $request)
    {
        RolHelper::validateAdminAccess();
        UserHelper::validateImportRequest($request);

        $file = $request->file('file');
        $spreadsheet = IOFactory::load($file->getRealPath());
        $sheet = $spreadsheet->getActiveSheet();
        $rows = $sheet->toArray();

        $header = array_map('strtoupper', $rows[0]);
        $required = ['NOMBRES', 'TIPO DE DOCUMENTO', 'DOCUMENTO', 'TELÉFONO', 'EMAIL', 'ROL'];
        foreach ($required as $col) {
            if (!in_array($col, $header)) {
                throw new Exception("Falta la columna $col");
            }
        }

        $maxRows = 500;
        if ($maxRows < count($rows) - 1) {
            throw new Exception("El archivo excede el máximo de $maxRows registros permitidos.");
        }

        $allowedRoles = ['DOCENTE', 'ESTUDIANTE'];
        $allowedDocumentTypes = ['DNI', 'CE', 'PASAPORTE'];
        $usersToMail = [];

        // Collect emails and documents for bulk validation
        $emails = [];
        $documents = [];
        foreach (array_slice($rows, 1) as $key => $row) {
            $data = array_combine($header, $row);

            // Validations
            if (!in_array(strtoupper($data['ROL']), $allowedRoles)) {
                throw new Exception("El rol '{$data['ROL']}' en la fila " . ($key + 2) . " no es válido. Los roles permitidos son: " . implode(', ', $allowedRoles));
            };

            if (!in_array(strtoupper($data['TIPO DE DOCUMENTO']), $allowedDocumentTypes)) {
                throw new Exception("El tipo de documento '{$data['TIPO DE DOCUMENTO']}' en la fila " . ($key + 2) . " no es válido. Los tipos permitidos son: " . implode(', ', $allowedDocumentTypes));
            };

            if (empty($data['NOMBRES']) || empty($data['TIPO DE DOCUMENTO']) || empty($data['DOCUMENTO']) || empty($data['EMAIL'])) {
                throw new Exception("Faltan datos obligatorios en la fila " . ($key + 2));
            };

            if (!filter_var($data['EMAIL'], FILTER_VALIDATE_EMAIL)) {
                throw new Exception("El email '{$data['EMAIL']}' en la fila " . ($key + 2) . " no es válido.");
            };

            if (!preg_match('/^\d{8,12}$/', $data['DOCUMENTO'])) {
                throw new Exception("El documento '{$data['DOCUMENTO']}' en la fila " . ($key + 2) . " no es válido. Debe tener entre 8 y 12 dígitos.");
            };

            // Duplicate validation in the file
            $emailKey = strtolower(trim($data['EMAIL']));
            $docKey = strtoupper(trim($data['TIPO DE DOCUMENTO'])) . '-' . trim($data['DOCUMENTO']);
            if (in_array($emailKey, $emails)) {
                throw new Exception("El email '{$data['EMAIL']}' está duplicado en el archivo (fila " . ($key + 2) . ").");
            }
            if (in_array($docKey, $documents)) {
                throw new Exception("El documento '{$data['TIPO DE DOCUMENTO']} {$data['DOCUMENTO']}' está duplicado en el archivo (fila " . ($key + 2) . ").");
            }
            $emails[] = $emailKey;
            $documents[] = $docKey;
        }

        // Duplicate validation in the database
        $existingEmails = User::whereIn('email', $emails)->pluck('email')->map(function ($e) {
            return strtolower($e);
        })->toArray();
        if (!empty($existingEmails)) {
            throw new Exception("Ya existen usuarios con los siguientes correos: " . implode(', ', $existingEmails));
        }
        $existingDocuments = Person::whereIn('document_type', array_map(function ($doc) {
            return explode('-', $doc)[0];
        }, $documents))
            ->whereIn('document_number', array_map(function ($doc) {
                return explode('-', $doc)[1];
            }, $documents))
            ->get(['document_type', 'document_number'])
            ->map(function ($p) {
                return strtoupper($p->document_type) . '-' . $p->document_number;
            })->toArray();
        if (!empty($existingDocuments)) {
            throw new Exception("Ya existen usuarios con los siguientes documentos: " . implode(', ', $existingDocuments));
        }

        $institutionName = SystemConfigurationHelper::getValueByKey('application_name');
        $roles = Rol::whereIn('name', $allowedRoles)->pluck('id', 'name');
        foreach (array_slice($rows, 1) as $key => $row) {
            $data = array_combine($header, $row);

            if (empty($data['ROL']) || !isset($roles[strtoupper($data['ROL'])])) {
                throw new Exception("El rol '{$data['ROL']}' en la fila " . ($key + 2) . " no es válido.");
            }

            // Create user and person
            $person = Person::create([
                'names' => $data['NOMBRES'],
                'document_type' => $data['TIPO DE DOCUMENTO'],
                'document_number' => $data['DOCUMENTO'],
                'phone' => $data['TELÉFONO'],
                'email' => $data['EMAIL'],
            ]);

            $user = User::create([
                'person_id' => $person->id,
                'email' => $data['EMAIL'],
                'password' => bcrypt($data['DOCUMENTO']),
                'rol' => $roles[strtoupper($data['ROL'])],
            ]);
            $user->roles()->attach($roles[strtoupper($data['ROL'])]);

            $usersToMail[] = [
                'subject' => 'Bienvenido a la plataforma',
                'to' => $user->email,
                'cc' => [],
                'institutionName' => $institutionName,
                'userName' => $person->names,
                'email' => $user->email,
                'loginUrl' => config('app.front_url') . '/login',
                'password' => $data['DOCUMENTO'],
            ];
        }

        // Send email notifications
        foreach (array_chunk($usersToMail, 50) as $chunk) {
            foreach ($chunk as $userData) {
                self::notifyUsersCredentials($userData);
            }
        }

        return 'Usuarios importados. Los usuarios recibirán un correo en los próximos minutos.';
    }

    private static function notifyUsersCredentials($data): void
    {
        $subject = $data['subject'];
        $to = $data['to'];
        $cc = $data['cc'] ?? [];
        $institutionName = $data['institutionName'];
        $userName = $data['userName'];
        $email = $data['email'];
        $loginUrl = $data['loginUrl'];
        $password = $data['password'];
        $body = EmailBodyTemplate::userCredentials;
        $body = str_replace('{{institutionName}}', $institutionName, $body);
        $body = str_replace('{{userName}}', $userName, $body);
        $body = str_replace('{{email}}', $email, $body);
        $body = str_replace('{{loginUrl}}', $loginUrl, $body);
        $body = str_replace('{{password}}', $password, $body);

        $data = (object) [
            'subject' => $subject,
            'body' => $body,
            'to' => $to,
            'cc' => $cc,
        ];

        MailerService::send($data);
    }
}
